// AI Form Filler - Background Service Worker
// LLM-first: sends raw form HTML to Gemini, uses executeScript for all-frames support

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'FILL_FORMS') {
    handleFillForms()
      .then(result => sendResponse(result))
      .catch(err => sendResponse({ success: false, error: err.message }));
    return true;
  }
});

// ============================================================
// ORCHESTRATION
// ============================================================

async function handleFillForms() {
  const settings = await getSettings();

  if (!settings.apiKey) return { success: false, error: 'API key not configured.' };
  if (!settings.isActive) return { success: false, error: 'Extension is not active. Toggle it on first.' };
  if (!settings.knowledgeBase?.trim()) return { success: false, error: 'Knowledge base is empty.' };

  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab) return { success: false, error: 'No active tab found.' };

  // Step 1: Extract forms from ALL frames using executeScript
  sendStatus('Extracting forms from all frames...', 'info');

  let formData;
  try {
    formData = await extractFormsFromAllFrames(tab.id);
  } catch (err) {
    return { success: false, error: 'Cannot access this page: ' + err.message };
  }

  if (!formData.formHtmlChunks.length && !formData.floatingFieldsHtml) {
    return { success: false, error: 'No forms found on this page (checked all frames).' };
  }

  sendStatus(`Found ${formData.formHtmlChunks.length} form(s). Calling Gemini AI...`, 'info');

  // Step 2: Send HTML + knowledge base to Gemini
  let fieldValues;
  try {
    fieldValues = await callGeminiWithHtml(settings, formData);
  } catch (err) {
    return { success: false, error: `Gemini API: ${err.message}` };
  }

  if (!fieldValues || Object.keys(fieldValues).length === 0) {
    return { success: false, error: 'Gemini returned no values to fill.' };
  }

  // Step 3: Fill fields across ALL frames using executeScript
  sendStatus(`Filling ${Object.keys(fieldValues).length} field(s)...`, 'info');

  let totalFilled;
  try {
    totalFilled = await fillFieldsInAllFrames(tab.id, fieldValues);
  } catch (err) {
    return { success: false, error: 'Failed to fill: ' + err.message };
  }

  return { success: true, filledCount: totalFilled };
}

// ============================================================
// CROSS-FRAME EXTRACTION via chrome.scripting.executeScript
// Returns aggregated form HTML from top frame + all iframes
// ============================================================

async function extractFormsFromAllFrames(tabId) {
  const results = await chrome.scripting.executeScript({
    target: { tabId, allFrames: true },
    func: extractFormsFn
  });

  // Aggregate results from every frame
  const aggregated = {
    url: '',
    title: '',
    formHtmlChunks: [],
    floatingFieldsHtml: ''
  };

  const floatingParts = [];

  for (const r of results) {
    if (!r.result) continue;
    const data = r.result;

    // Use top frame's URL/title (frameId 0)
    if (r.frameId === 0) {
      aggregated.url = data.url;
      aggregated.title = data.title;
    }

    if (data.formHtmlChunks?.length) {
      for (const chunk of data.formHtmlChunks) {
        aggregated.formHtmlChunks.push(chunk);
      }
    }

    if (data.floatingFieldsHtml) {
      floatingParts.push(data.floatingFieldsHtml);
    }
  }

  aggregated.floatingFieldsHtml = floatingParts.join('\n');

  // Fallback: if no URL captured from frame 0
  if (!aggregated.url) {
    try {
      const tabInfo = await chrome.tabs.get(tabId);
      aggregated.url = tabInfo.url || '';
      aggregated.title = tabInfo.title || '';
    } catch {}
  }

  console.log(`[AI Form Filler] Total forms from all frames: ${aggregated.formHtmlChunks.length}`);
  return aggregated;
}

// Self-contained extraction function injected into each frame
function extractFormsFn() {
  const result = {
    url: location.href,
    title: document.title,
    formHtmlChunks: [],
    floatingFieldsHtml: ''
  };

  // Extract <form> elements
  const forms = document.querySelectorAll('form');
  for (const form of forms) {
    const html = compactForm(form);
    if (html) result.formHtmlChunks.push(html);
  }

  // Orphaned fields not inside <form>
  const allFields = document.querySelectorAll('input, textarea, select');
  const orphans = [];
  for (const el of allFields) {
    if (!el.closest('form')) orphans.push(el);
  }
  if (orphans.length > 0) {
    const parts = [];
    for (const el of orphans) {
      if (el.type === 'hidden') continue;
      // Include label if found
      if (el.id) {
        const lbl = document.querySelector('label[for="' + el.id + '"]');
        if (lbl) parts.push(lbl.outerHTML);
      }
      parts.push(el.outerHTML);
    }
    result.floatingFieldsHtml = parts.join('\n');
  }

  // Shadow DOM forms
  try {
    const walker = document.querySelectorAll('*');
    for (const node of walker) {
      if (node.shadowRoot) {
        const sForms = node.shadowRoot.querySelectorAll('form');
        for (const f of sForms) {
          const html = compactForm(f);
          if (html) result.formHtmlChunks.push(html);
        }
      }
    }
  } catch (e) {}

  return result;

  function compactForm(form) {
    const clone = form.cloneNode(true);
    // Remove non-essential elements
    clone.querySelectorAll('script, style, noscript, svg, img').forEach(el => el.remove());

    // Compact long selects: keep first 10 + last 5, note total
    clone.querySelectorAll('select').forEach(sel => {
      const opts = sel.querySelectorAll('option');
      if (opts.length > 20) {
        const total = opts.length;
        const keep = new Set();
        for (let i = 0; i < 10 && i < total; i++) keep.add(opts[i]);
        for (let i = total - 5; i < total; i++) { if (i >= 0) keep.add(opts[i]); }
        let markerPlaced = false;
        for (let i = 0; i < total; i++) {
          if (!keep.has(opts[i])) {
            if (!markerPlaced) {
              const m = document.createComment(' ... ' + (total - 15) + ' more options ... ');
              opts[i].parentNode.insertBefore(m, opts[i]);
              markerPlaced = true;
            }
            opts[i].remove();
          }
        }
      }
    });

    // Strip decorative attributes
    clone.querySelectorAll('*').forEach(el => {
      el.removeAttribute('style');
      el.removeAttribute('data-hint');
      el.removeAttribute('data-analytics');
    });

    const html = clone.outerHTML;
    // Skip if no fillable fields
    if (!/<(input|textarea|select)\b/i.test(html)) return null;
    // Cap at 30KB per form
    return html.length > 30000 ? html.substring(0, 30000) + '\n<!-- truncated -->' : html;
  }
}

// ============================================================
// CROSS-FRAME FILLING via chrome.scripting.executeScript
// ============================================================

async function fillFieldsInAllFrames(tabId, fieldValues) {
  const results = await chrome.scripting.executeScript({
    target: { tabId, allFrames: true },
    func: fillFieldsFn,
    args: [fieldValues]
  });

  let totalFilled = 0;
  for (const r of results) {
    if (r.result && typeof r.result.filledCount === 'number') {
      totalFilled += r.result.filledCount;
    }
  }
  return totalFilled;
}

// Self-contained fill function injected into each frame
function fillFieldsFn(values) {
  let filledCount = 0;

  for (const [key, value] of Object.entries(values)) {
    if (value === '' || value === null || value === undefined) continue;
    if (value === 'SKIP' || value === '__skip__') continue;

    const el = findEl(key);
    if (!el) continue;

    try {
      if (fillEl(el, value)) filledCount++;
    } catch (e) {
      console.warn('[AI Form Filler] Fill error:', key, e);
    }
  }

  return { filledCount };

  function findEl(key) {
    // 1. By ID
    let el = document.getElementById(key);
    if (el) return el;
    // 2. By name
    el = document.querySelector('[name="' + key.replace(/"/g, '\\"') + '"]');
    if (el) return el;
    // 3. Prefixed: "id:xxx", "name:xxx"
    if (key.startsWith('id:')) return document.getElementById(key.substring(3));
    if (key.startsWith('name:')) return document.querySelector('[name="' + key.substring(5).replace(/"/g, '\\"') + '"]');
    // 4. CSS selector
    if (key.startsWith('#') || key.startsWith('.') || key.startsWith('[')) {
      try { return document.querySelector(key); } catch { return null; }
    }
    return null;
  }

  function fillEl(element, value) {
    const tag = element.tagName.toLowerCase();
    const type = (element.type || '').toLowerCase();

    if (element.getAttribute('contenteditable') === 'true' && tag !== 'input') {
      element.focus();
      element.innerHTML = '';
      document.execCommand('selectAll', false, null);
      document.execCommand('insertText', false, String(value));
      element.dispatchEvent(new Event('input', { bubbles: true }));
      element.dispatchEvent(new Event('change', { bubbles: true }));
      return true;
    }

    if (tag === 'select') return fillSelect(element, value);
    if (type === 'checkbox') return fillCheckbox(element, value);
    if (type === 'radio') return fillRadio(element, value);
    return fillText(element, value);
  }

  function fillText(element, value) {
    const str = String(value);
    const tag = element.tagName.toLowerCase();
    element.scrollIntoView({ block: 'nearest', behavior: 'instant' });
    element.focus();
    element.dispatchEvent(new FocusEvent('focus', { bubbles: true }));

    // Native setter bypasses React/Angular/Vue interception
    const proto = tag === 'textarea' ? HTMLTextAreaElement.prototype : HTMLInputElement.prototype;
    const setter = Object.getOwnPropertyDescriptor(proto, 'value')?.set;
    if (setter) { setter.call(element, ''); }
    element.dispatchEvent(new Event('input', { bubbles: true }));
    if (setter) { setter.call(element, str); } else { element.value = str; }

    element.dispatchEvent(new Event('input', { bubbles: true, cancelable: true }));
    element.dispatchEvent(new Event('change', { bubbles: true, cancelable: true }));
    element.dispatchEvent(new InputEvent('input', { bubbles: true, inputType: 'insertText', data: str }));
    element.dispatchEvent(new KeyboardEvent('keydown', { bubbles: true, key: 'a' }));
    element.dispatchEvent(new KeyboardEvent('keyup', { bubbles: true, key: 'a' }));
    element.dispatchEvent(new FocusEvent('blur', { bubbles: true }));
    return true;
  }

  function fillSelect(element, value) {
    const str = String(value).trim();
    const low = str.toLowerCase();
    const opts = Array.from(element.options);

    const match =
      opts.find(o => o.value === str) ||
      opts.find(o => o.value.toLowerCase() === low) ||
      opts.find(o => o.textContent.trim().toLowerCase() === low) ||
      opts.find(o => o.textContent.trim().toLowerCase().startsWith(low) && o.value && o.value !== 'None') ||
      opts.find(o => (o.textContent.trim().toLowerCase().includes(low) || low.includes(o.textContent.trim().toLowerCase())) && o.value && o.value !== 'None');

    if (!match) return false;

    const setter = Object.getOwnPropertyDescriptor(HTMLSelectElement.prototype, 'value')?.set;
    if (setter) { setter.call(element, match.value); } else { element.value = match.value; }
    element.dispatchEvent(new Event('input', { bubbles: true }));
    element.dispatchEvent(new Event('change', { bubbles: true }));
    return true;
  }

  function fillCheckbox(element, value) {
    const want = value === true || value === 'true' || value === 'yes' || value === '1' || value === 'on';
    if (element.checked !== want) element.click();
    return true;
  }

  function fillRadio(element, value) {
    const str = String(value).trim();
    const low = str.toLowerCase();
    if (element.value === str || element.value.toLowerCase() === low) { element.click(); return true; }
    if (element.name) {
      const radios = document.querySelectorAll('input[type="radio"][name="' + element.name.replace(/"/g, '\\"') + '"]');
      for (const r of radios) {
        if (r.value === str || r.value.toLowerCase() === low) { r.click(); return true; }
      }
    }
    return false;
  }
}

// ============================================================
// GEMINI API
// ============================================================

async function callGeminiWithHtml(settings, formData) {
  const { apiKey, model, knowledgeBase, defaultEmail, defaultPassword } = settings;
  const modelName = model || 'gemini-2.5-flash';
  const endpoint = `https://generativelanguage.googleapis.com/v1beta/models/${modelName}:generateContent?key=${apiKey}`;

  const body = {
    contents: [{ role: 'user', parts: [{ text: buildUserPrompt(knowledgeBase, defaultEmail, defaultPassword, formData) }] }],
    systemInstruction: { parts: [{ text: buildSystemPrompt() }] },
    generationConfig: {
      responseMimeType: 'application/json',
      temperature: 0.1,
      maxOutputTokens: 8192
    }
  };

  const response = await fetchRetry(endpoint, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(body)
  });

  if (!response.ok) {
    let msg = `HTTP ${response.status}`;
    try { msg = (await response.json()).error?.message || msg; } catch {}
    throw new Error(msg);
  }

  const data = await response.json();
  const text = data.candidates?.[0]?.content?.parts?.[0]?.text;
  if (!text) {
    const reason = data.candidates?.[0]?.finishReason;
    throw new Error(reason === 'SAFETY' ? 'Blocked by safety filters.' : 'Empty response from Gemini.');
  }
  return parseJson(text);
}

function buildSystemPrompt() {
  return `You are an expert form-filling AI. You receive the RAW HTML of web form(s) and a knowledge base. Analyze the HTML, understand each field, and return values to fill.

RULES:
1. Read the HTML carefully. Identify each fillable field by its "id" or "name" attribute.
2. SKIP: type="hidden" fields, submit buttons, CSRF tokens (names like syx_*), timezone fields, honeypot traps (fields with class="hidden" on type="text", or suspicious names).
3. Use <label>, placeholder, aria-label, and surrounding HTML context to understand each field.
4. For <select>: return the EXACT option "value" attribute (e.g., "IN" not "India"). Read the <option value="XX"> tags.
5. For checkboxes: return true/false.
6. For radio buttons: return the matching option's value attribute.
7. For dates: YYYY-MM-DD format.
8. For textareas (comments, inquiries, descriptions, bios): compose natural, substantive text from the knowledge base.
9. For email fields: use the default email.
10. For password fields: use the default password.
11. For URL/website fields: use relevant URLs from the knowledge base.
12. For name fields: extract from knowledge base.
13. For country selects: return the 2-letter ISO code (the option value).
14. For number/donation/amount fields: use "0" or an appropriate value.
15. Omit fields you cannot fill.

OUTPUT: JSON object where each key is the field's "id" (preferred) or "name", each value is the string/boolean to fill.

Example:
{
  "u_abc_123": "John",
  "u_abc_456": "john@example.com",
  "u_abc_789": "IN",
  "u_abc_012": "I am interested in..."
}`;
}

function buildUserPrompt(knowledgeBase, defaultEmail, defaultPassword, formData) {
  let html = formData.formHtmlChunks.join('\n\n---NEXT FORM---\n\n');
  if (formData.floatingFieldsHtml) html += '\n\n---FLOATING FIELDS---\n\n' + formData.floatingFieldsHtml;

  return `=== PAGE ===
URL: ${formData.url}
Title: ${formData.title}

=== FORM HTML ===
${html}

=== KNOWLEDGE BASE ===
${knowledgeBase}

=== DEFAULT CREDENTIALS ===
Email: ${defaultEmail || '(not set)'}
Password: ${defaultPassword || '(not set)'}

=== TASK ===
Analyze the HTML. Return a JSON mapping field IDs/names to values. Skip hidden fields, honeypots, CSRF tokens, and submit buttons.`;
}

// ============================================================
// HELPERS
// ============================================================

function parseJson(text) {
  try { return JSON.parse(text); } catch {}
  const cb = text.match(/```(?:json)?\s*([\s\S]*?)```/);
  if (cb) try { return JSON.parse(cb[1].trim()); } catch {}
  const obj = text.match(/\{[\s\S]*\}/);
  if (obj) try { return JSON.parse(obj[0]); } catch {}
  throw new Error('Failed to parse Gemini response as JSON.');
}

async function fetchRetry(url, options, retries = 2) {
  for (let i = 0; i <= retries; i++) {
    try {
      const r = await fetch(url, options);
      if ((r.status === 429 || r.status === 503) && i < retries) { await sleep(1000 * 2 ** i); continue; }
      return r;
    } catch (err) {
      if (i === retries) throw err;
      await sleep(1000 * 2 ** i);
    }
  }
}

function sleep(ms) { return new Promise(r => setTimeout(r, ms)); }

function sendStatus(text, type) {
  chrome.runtime.sendMessage({ action: 'FILL_STATUS', text, type }).catch(() => {});
}

function getSettings() {
  return new Promise(resolve => {
    chrome.storage.local.get(['apiKey', 'model', 'defaultEmail', 'defaultPassword', 'knowledgeBase', 'isActive'], resolve);
  });
}
